importScripts('https://storage.googleapis.com/workbox-cdn/releases/5.1.4/workbox-sw.js');

const CACHE = "pwa-cache-v38";
const offlineFallbackPage = "offline/";  // Ensure this points to your actual offline page
const filesToCache = [
  'offline/',
  'assets/files/backgrounds/error_page_bg.jpg',
  'assets/files/backgrounds/offline_error_expression_text_bg.jpg',
  'assets/thirdparty/bootstrap/bootstrap.min.css',
  'assets/css/error_page/error_page.css',
  'assets/files/defaults/favicon.png',
  'assets/fonts/montserrat/montserrat-bold.woff',
  'assets/fonts/montserrat/montserrat-medium.woff',
  'assets/fonts/montserrat/montserrat-semibold.woff',
  'assets/fonts/montserrat/font.css',
  'assets/thirdparty/bootstrap/bootstrap.bundle.min.js',
];

// Message handler to skip waiting phase
self.addEventListener("message", (event) => {
  if (event.data && event.data.type === "SKIP_WAITING") {
    self.skipWaiting();
  }
});

// Install event to cache files
self.addEventListener('install', async (event) => {
  event.waitUntil(
    caches.open(CACHE)
      .then((cache) => {
        return cache.addAll(filesToCache);
      })
  );
});

async function updateUnreadCount() {
  try {
    const dataToSend = new FormData();
    dataToSend.append('realtime', 'true');
    dataToSend.append('unread_group_messages', 0);
    dataToSend.append('unread_private_chat_messages', 0);
    dataToSend.append('unread_site_notifications', 0);

    const response = await fetch('web_request/', {
      method: 'POST',
      body: dataToSend,
      credentials: 'include',
    });

    if (!response.ok) {
      throw new Error('Network response was not ok');
    }

    const data = await response.json();
    var unread_count = 0;
    if (data?.unread_private_chat_messages) {
      unread_count += Number(data.unread_private_chat_messages) || 0;
    }
    if (data?.unread_site_notifications) {
      unread_count += Number(data.unread_site_notifications) || 0;
    }
    
    if (data?.unread_group_messages) {
      unread_count += Number(data.unread_group_messages) || 0;
    }
    if (unread_count !== 0) {
      setBadge(unread_count);
    }
  } catch (err) {
    console.error('Failed to fetch unread count', err);
  }
}

//setInterval(updateUnreadCount, 10000);

async function setBadge(count) {
  // If the platform supports it, set the badge.
  if ('setAppBadge' in navigator) {
    try {
      await navigator.setAppBadge(count);
      console.log('Badge set to:', count);
    } catch (err) {
      console.error('Failed to set app badge:', err);
    }
  } else {
    console.log('App Badge API is not supported on this platform.');
  }
}

// Enable navigation preload if supported
if (workbox.navigationPreload.isSupported()) {
  workbox.navigationPreload.enable();
}

// Fetch event to respond with cached or network data
self.addEventListener('fetch', (event) => {
  const pwrequestUrl = new URL(event.request.url);

  // Check if the request should be handled by the cache
  if (!filesToCache.some(file => pwrequestUrl.pathname.endsWith(file))) {
    return;
  }

  // Ignore certain requests
  if (event.request.url.endsWith('realtime_request/') || event.request.url.endsWith('web_request/')) {
    return;
  }

  event.respondWith((async () => {
    try {
      // First try to get the preloaded response
      const preloadResp = await event.preloadResponse;
      if (preloadResp) {
        return preloadResp;  // Return the preloaded response if available
      }

      // Try to fetch from the network
      const networkResp = await fetch(event.request);
      // Optionally cache the network response for future requests
      const cache = await caches.open(CACHE);
      cache.put(event.request, networkResp.clone());  // Cache the cloned response
      return networkResp;  // Return the network response
    } catch (error) {
      // On failure, return the offline fallback page if available
      const cache = await caches.open(CACHE);
      const cachedResp = await cache.match(offlineFallbackPage);
      return cachedResp || new Response('Offline', { status: 404 });  // Return a 404 if offline page is not found
    }
  })());
});
