<?php
namespace WeDevs\PM_Pro\Modules\Sprint\Src\Helper;

use WP_REST_Request;
use WeDevs\PM\task\Helper\Task;
use WeDevs\PM\Task\Controllers\Task_Controller;
use WeDevs\PM\Project\Helper\Project;
// data: {
//  with: 'projects',
//  per_page: '10',
//  select: 'id, title',
//  id: [1,2],
//  title: 'Rocket', 'test'
//  page: 1,
//  orderby: 'title:asc,id:desc'
//  sprint_meta: 'total_sprints,total_tasks,total_complete_tasks,total_incomplete_tasks,total_discussion_boards,total_milestones,total_comments,total_files,total_activities'
// },

class Sprint {
    private static $_instance;
    private $query_params;
    private $select;
    private $join;
    private $where;
    private $limit;
    private $orderby;
    private $with = ['creator'];
    private $sprints;
    private $sprint_ids;
    private $is_single_query = false;
    private $tb_project;
    private $tb_sprint;
    private $tb_task;
    private $tb_project_user;
    private $tb_task_user;
    private $tb_categories;
    private $tb_category_project;
    private $found_rows;

    public static function getInstance() {
        // if ( !self::$_instance ) {
        //     self::$_instance = new self();
        // }

        return new self();
    }

    function __construct() {
        $this->set_table_name();
    }

    public static function get_sprints( WP_REST_Request $request ) {
        $sprints = self::get_results( $request->get_params() );

        wp_send_json( $sprints );
    }

    public static function get_results( $params ) {
        $self = self::getInstance();
        $self->query_params = $params;

        $self->select()
            ->join()
            ->where()
            ->limit()
            ->orderby()
            ->get()
            ->with()
            ->meta();

        $response = $self->format_sprints( $self->sprints );

        if ( wedevs_pm_is_single_query( $params ) ) {
            return ['data' => $response['data'][0]] ;
        }

        return $response;
    }

    /**
     * Format sprint data
     *
     * @param array $sprints
     *
     * @return array
     */
    public function format_sprints( $sprints ) {
        $response = [
            'data' => [],
            'meta' => []
        ];

        if ( ! is_array( $sprints ) ) {
            $response['data'] = $this->fromat_sprint( $sprints );

            return $response;
        }

        foreach ( $sprints as $key => $sprint ) {
            $sprints[$key] = $this->fromat_sprint( $sprint );
        }

        $response['data']  = $sprints;
        $response ['meta'] = $this->set_sprint_meta();

        return $response;
    }

    /**
     * Set meta data
     */
    private function set_sprint_meta() {
        return [
            'pagination'    => [
                "per_page"    => $this->get_per_page(),
                "total_pages" => ceil( $this->found_rows/$this->get_per_page() ),
                "found_rows"  => $this->found_rows,
            ]
        ];
    }

    public function fromat_sprint( $sprint ) {

        $items = [
            'id'          => (int) $sprint->id,
            'title'       => isset( $sprint->title ) ? (string) $sprint->title : null,
            'description' => isset( $sprint->description ) ? wedevs_pm_filter_content_url( $sprint->description ) : null,
            'order'       => isset( $sprint->order ) ? (int) $sprint->order : null,
            'status'      => isset( $sprint->status ) ? $sprint->status : null,
            'start_at'    => isset( $sprint->start_at ) ? $sprint->start_at : null,
            'due_date'    => isset( $sprint->due_date ) ? $sprint->due_date : null,
            'created_at'  => isset( $sprint->status ) ?  wedevs_pm_format_date( $sprint->created_at ) : null,
            'meta'        => empty( $sprint->meta ) ? [] : $sprint->meta
        ];

        $items = $this->item_with( $items, $sprint );

        return apply_filters( 'wedevs_pm_sprint_transform', $items, $sprint );
    }

    private function item_with( $items, $sprint ) {
        $with = empty( $this->query_params['with'] ) ? [] : $this->query_params['with'];

        if ( ! is_array( $with ) ) {
            $with = explode( ',', str_replace(' ', '', $with ) );
        }

        $with = array_merge( $this->with, $with );

        $sprint_with_items =  array_intersect_key( (array) $sprint, array_flip( $with ) );

        $items = array_merge( $items, $sprint_with_items );

        return $items;
    }

    private function with() {
        $this->creator()
            ->include_projects()
            ->include_incomplete_tasks()
            ->include_complete_tasks();

        return $this;
    }

    private function creator() {

        if ( empty( $this->sprints ) ) {
            return $this;
        }

        $creator_ids = wp_list_pluck( $this->sprints, 'created_by' );
        $creator_ids = array_unique( $creator_ids );

        $creators = wedevs_pm_get_users( [ 'id' => $creator_ids ] );
        $creators = $creators['data'];

        $items = [];

        foreach ( $creators as $key => $creator ) {
            $items[$creator['id']] = $creator;
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $s_creator = empty( $items[$sprint->created_by] ) ? [] : $items[$sprint->created_by];

            $sprint->creator = [ 'data' => $s_creator ];
        }

        return $this;
    }

        /**
     * Set project ssignees
     *
     * @return class object
     */
    private function include_projects() {
        global $wpdb;
        $with = empty( $this->query_params['with'] ) ? [] : $this->query_params['with'];

        if ( ! is_array( $with ) ) {
            $with = explode( ',', str_replace(' ', '', $with ) );
        }

        if ( ! in_array( 'projects', $with ) || empty( $this->sprint_ids ) ) {
            return $this;
        }

        $tb_projects   = wedevs_pm_tb_prefix() . 'pm_projects';
        $tb_sp_pro     = wedevs_pm_tb_prefix() . 'pm_sprint_projects';
        $sprint_format = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data    = $this->sprint_ids;

        $query = "SELECT DISTINCT pj.id as project_id, pj.title, sp.sprint_id
            FROM $tb_projects as pj
            LEFT JOIN $tb_sp_pro as sp ON sp.project_id = pj.id
            where sp.sprint_id IN ($sprint_format)";

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );
        $project_ids = wp_list_pluck( $results, 'project_id' );

        $db_projects = Project::get_results([
            'id' => $project_ids,
            'with' => ['assignees']
        ]);

        $projects = [];

        foreach ( $results as $key => $result ) {
            foreach ( $db_projects['data'] as $key => $project ) {
                if ( $project['id'] == $result->project_id ) {
                    $project['project_id'] = $project['id'];
                    $projects[$result->sprint_id][] = $project;
                }
            }
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $sprint->projects['data'] = empty( $projects[$sprint->id] ) ? [] : $projects[$sprint->id];
        }

        return $this;
    }

    private function meta() {

        $this->get_estimation();

        if ( wedevs_pm_get_estimation_type() == 'task' ) {
            $this->get_users_meta();
        }

        if ( wedevs_pm_get_estimation_type() == 'subtask' ) {
            $this->get_subtask_users_meta();
        }

        return $this;
    }

    private function get_estimation() {
        global $wpdb;

        if ( empty( $this->sprint_ids ) ) {
            return $this;
        }

        $sprint_ids = is_array( $this->sprint_ids ) ? $this->sprint_ids : [$sprint_ids];
        $sprint_ids = implode( ',', $sprint_ids );
        $sprint_project_tasks = $wpdb->prefix . 'pm_sprint_project_tasks';
        $tb_tasks = $wpdb->prefix . 'pm_tasks';

        $query = "SELECT DISTINCT tsk.estimation, tsk.status, spk.sprint_id, tsk.id as task_id
            FROM {$tb_tasks} as tsk
            LEFT JOIN {$sprint_project_tasks} as spk ON spk.task_id=tsk.id
            WHERE spk.sprint_id IN ($sprint_ids)";

        $results = $wpdb->get_results( $query );
        $sprints = [];
        $tasks = [];

        foreach ( $results as $key => $task ) {
            $tasks[$task->task_id] = $task;
        }

        if ( wedevs_pm_get_estimation_type() == 'subtask' ) {
            $task_ids = wp_list_pluck( $results, 'task_id' );
            $task_ids = empty( $task_ids ) ? [0] : $task_ids;
            $task_ids = array_unique( $task_ids );
            $task_ids = implode( ',', $task_ids );
            $sub_est  = [];

            $task_sprints = wp_list_pluck( $results, 'sprint_id', 'task_id' );

            if ( ! empty( $task_ids ) ) {
                $query = "SELECT DISTINCT tsk.estimation, tsk.status, tsk.parent_id as task_id, spt.sprint_id, tsk.id as subtask_id
                FROM {$tb_tasks} as tsk
                    LEFT JOIN {$sprint_project_tasks} as spt ON spt.task_id=tsk.parent_id
                WHERE tsk.parent_id IN ($task_ids)";

                $results = $wpdb->get_results( $query );

                foreach ( $results as $key => $subtask ) {
                    if ( $tasks[$subtask->task_id]->status == 0 ) {
                        continue;
                    }
                    $subtask->status = $tasks[$subtask->task_id]->status;
                }
            }
        }


        foreach ( $results as $key => $result ) {
            if ( $result->status == 1 ) {
                $sprints[$result->sprint_id]['completed'][] = $result;
            }

            if ( $result->status == 0 ) {
                $sprints[$result->sprint_id]['incomplete'][] = $result;
            }

        }

        $return = [];

        foreach ( $sprints as $sprint_id => $sprint ) {

            $completed              = empty( $sprint['completed'] ) ? [] : $sprint['completed'];
            $incomplete             = empty( $sprint['incomplete'] ) ? [] : $sprint['incomplete'];

            $estimation_completed   = wp_list_pluck( $completed, 'estimation' );
            $estimation_incomplete  = wp_list_pluck( $incomplete, 'estimation' );

            $total_tasks            = count( $completed ) + count( $incomplete );

            $total_estimation_time  = array_sum( $estimation_completed ) + array_sum( $estimation_incomplete );
            $total_completed_time   = array_sum( $estimation_completed );
            $total_incomplete_time  = array_sum( $estimation_incomplete );

            if ( empty( $total_estimation_time ) ) {
                $progress = 0;
            } else {
                $progress = ($total_completed_time * 100)/$total_estimation_time;
            }

            $return[$sprint_id] = [
                'total_estimated_time'   => $total_estimation_time,
                'total_completed_time'   => $total_completed_time,
                'total_incomplete_time'  => $total_incomplete_time,
                'progress'               => round( $progress ),
            ];
        }

        foreach ( $this->sprints as $key => $sprint ) {
            if ( empty( $return[$sprint->id] ) ) {
                $sprint->meta = [
                    'total_estimated_time'   => 0,
                    'total_completed_time'   => 0,
                    'progress'               => 0,
                ];
            } else {
               $sprint->meta = $return[$sprint->id];
            }
        }

        return $this;
    }

    private function get_subtask_users_meta() {
        global $wpdb;

        if ( empty( $this->sprint_ids ) ) {
            return $this;
        }

        $sprint_project_tasks = $wpdb->prefix . 'pm_sprint_project_tasks';
        $tb_tasks                = $wpdb->prefix . 'pm_tasks';
        $user                 = $wpdb->base_prefix . 'users';
        $tb_user_meta         = $wpdb->base_prefix . 'usermeta';
        $task_users           = $wpdb->prefix . 'pm_assignees';

        $sprint_ids = is_array( $this->sprint_ids ) ? $this->sprint_ids : [$sprint_ids];
        $sprint_ids = implode( ',', $sprint_ids );

        $query = "SELECT DISTINCT tsk.id as task_id, tsk.status, spt.sprint_id
                FROM {$tb_tasks} as tsk
                LEFT JOIN {$sprint_project_tasks} as spt ON spt.task_id=tsk.id
                WHERE spt.sprint_id IN ($sprint_ids)";

        $results = $wpdb->get_results( $query );

        $tasks = [];

        foreach ( $results as $key => $task ) {
            $tasks[$task->task_id] = $task;
        }

        $sprint_tasks    = $wpdb->get_results( $query );
        $sprint_task_ids = wp_list_pluck( $sprint_tasks, 'task_id' );
        $sprint_task_ids = empty( $sprint_task_ids ) ? [0] : $sprint_task_ids;
        $sprint_task_ids = array_unique( $sprint_task_ids );
        $sprint_task_ids = implode( ',', $sprint_task_ids );

        if ( is_multisite() ) {
            $meta_key = wedevs_pm_user_meta_key();

            $query = "SELECT DISTINCT tsk.parent_id as task_id, tsk.status, usr.ID as user_id,
                usr.display_name, spt.sprint_id, tsk.estimation, tsk.id as subtask_id

                FROM {$tb_tasks} as tsk
                    LEFT JOIN {$sprint_project_tasks} as spt ON spt.task_id=tsk.parent_id
                    LEFT JOIN {$task_users} as tusr ON tusr.task_id=tks.id
                    LEFT JOIN {$user} as usr ON usr.ID=tusr.assigned_to
                    LEFT JOIN {$tb_user_meta} as umeta ON umeta.user_id = usr.ID

                WHERE spt.sprint_id IN ($sprint_ids)
                    AND umeta.meta_key='$meta_key'
                    AND tsk.parent_id IN ($sprint_task_ids)";
        } else {
            $query = "SELECT DISTINCT tsk.parent_id as task_id, tsk.status, usr.ID as user_id,
                usr.display_name, spt.sprint_id, tsk.estimation, tsk.id as subtask_id

                FROM {$tb_tasks} as tsk
                    LEFT JOIN {$sprint_project_tasks} as spt ON spt.task_id=tsk.parent_id
                    LEFT JOIN {$task_users} as tusr ON tusr.task_id=tsk.id
                    LEFT JOIN {$user} as usr ON usr.ID=tusr.assigned_to

                WHERE spt.sprint_id IN ($sprint_ids)
                    AND tsk.parent_id IN ($sprint_task_ids)";
        }

        $results = $wpdb->get_results( $query );

        foreach ( $results as $key => $subtask ) {
            if ( $tasks[$subtask->task_id]->status == 0 ) {
                continue;
            }
            $subtask->status = $tasks[$subtask->task_id]->status;
        }

        $estimation = [];

        foreach ( $results as $result ) {
            if ( $result->status == 0 && !empty( $result->user_id ) ) {
                $estimation[$result->sprint_id][$result->user_id]['incomplete'][] = $result->estimation;
            }

            if ( $result->status == 1 && !empty( $result->user_id ) ) {
                $estimation[$result->sprint_id][$result->user_id]['completed'][] = $result->estimation;
            }
        }

        $sprint_users = $this->get_sprints_users();

        foreach ( $this->sprints as $sprint ) {

            if ( empty( $sprint_users[$sprint->id] ) ) {
                $sprint->meta['users'] = [];

                continue;
            }

            foreach ( $sprint_users[$sprint->id] as $user_id => $user ) {

                if ( empty( $estimation[$sprint->id][$user_id] ) ) {
                    $sprint->meta['users'][] = [
                        'total_estimated_time'   => 0,
                        'total_completed_time'   => 0,
                        'total_incomplete_time'  => 0,
                        'progress'               => 0,
                        'avatar_url'             => get_avatar_url( $user_id ),
                        'display_name'           => $user->display_name,
                        'id'                     => $user_id

                    ];

                    continue;
                }

                if ( empty( $estimation[$sprint->id][$user_id]['completed'] ) ) {
                    $completed_estimation = 0;
                } else {
                    $completed_estimation = array_sum( $estimation[$sprint->id][$user_id]['completed'] );
                }

                if ( empty( $estimation[$sprint->id][$user_id]['incomplete'] ) ) {
                    $incomplete_estimation = 0;
                } else {
                    $incomplete_estimation = array_sum( $estimation[$sprint->id][$user_id]['incomplete'] );
                }

                $total_estimation_time = $completed_estimation + $incomplete_estimation;

                if ( empty( $total_estimation_time ) ) {
                    $progress = 0;
                } else {
                    $progress = ($completed_estimation * 100)/$total_estimation_time;
                }

                $sprint->meta['users'][] = [
                    'total_estimated_time'   => $total_estimation_time,
                    'total_completed_time'   => $completed_estimation,
                    'total_incomplete_time'  => $incomplete_estimation,
                    'progress'               => round( $progress ),
                    'avatar_url'             => get_avatar_url( $user_id ),
                    'display_name'           => $user->display_name,
                    'id'                     => $user_id
                ];
            }
        }

        return $this;

    }

    private function get_users_meta() {
        global $wpdb;

        if ( empty( $this->sprint_ids ) ) {
            return $this;
        }

        $sprint_project_tasks = $wpdb->prefix . 'pm_sprint_project_tasks';
        $sprint_projects      = $wpdb->prefix . 'pm_sprint_projects';
        $projects             = $wpdb->prefix . 'pm_projects';
        $tasks                = $wpdb->prefix . 'pm_tasks';
        $role_user            = $wpdb->prefix . 'pm_role_user';
        $user                 = $wpdb->base_prefix . 'users';
        $tb_user_meta         = $wpdb->base_prefix . 'usermeta';
        $sprint_projects      = $wpdb->prefix . 'pm_sprint_projects';
        $sprint               = $wpdb->prefix . 'pm_sprints';
        $task_users           = $wpdb->prefix . 'pm_assignees';

        $sprint_ids = is_array( $this->sprint_ids ) ? $this->sprint_ids : [$sprint_ids];
        $sprint_ids = implode( ',', $sprint_ids );

        if ( is_multisite() ) {
            $meta_key = wedevs_pm_user_meta_key();

            $query = "SELECT DISTINCT tsk.id as task_id, tsk.status, usr.ID as user_id, usr.display_name, spt.sprint_id, tsk.estimation
                FROM {$tasks} as tsk
                LEFT JOIN {$sprint_project_tasks} as spt ON spt.task_id=tsk.id
                LEFT JOIN {$task_users} as tusr ON tusr.task_id=spt.task_id
                LEFT JOIN {$user} as usr ON usr.ID=tusr.assigned_to
                LEFT JOIN {$tb_user_meta} as umeta ON umeta.user_id = usr.ID
                WHERE spt.sprint_id IN ($sprint_ids) AND umeta.meta_key='$meta_key'";
        } else {
            $query = "SELECT DISTINCT tsk.id as task_id, tsk.status, usr.ID as user_id, usr.display_name, spt.sprint_id, tsk.estimation
                FROM {$tasks} as tsk
                LEFT JOIN {$sprint_project_tasks} as spt ON spt.task_id=tsk.id
                LEFT JOIN {$task_users} as tusr ON tusr.task_id=spt.task_id
                LEFT JOIN {$user} as usr ON usr.ID=tusr.assigned_to
                WHERE spt.sprint_id IN ($sprint_ids)";
        }

        $results = $wpdb->get_results( $query );
        $sprints = [];
        $users = [];

        foreach ( $results as $result ) {
            if ( $result->status == 0 && !empty( $result->user_id ) ) {
                $sprints[$result->sprint_id][$result->user_id]['incomplete'][] = $result;
            }

            if ( $result->status == 1 && !empty( $result->user_id ) ) {
                $sprints[$result->sprint_id][$result->user_id]['completed'][] = $result;
            }

            if ( !empty( $result->user_id ) ) {
                $users[$result->user_id] = $result;
            }

        }

        foreach ( $this->sprints as $sprint ) {

            if ( empty( $sprints[$sprint->id] ) ) {
                $sprint->meta['users'] = [];

                continue;
            }

            foreach ( $sprints[$sprint->id] as $user_id => $user_task ) {

                if ( ! empty( $user_task['completed'] ) ) {
                    $completed_estimation = array_sum( wp_list_pluck( $user_task['completed'], 'estimation' ) );
                } else {
                    $completed_estimation = 0;
                    $total_completed_tasks = 0;
                }

                if ( ! empty( $user_task['incomplete'] ) ) {
                    $incomplete_estimation = array_sum( wp_list_pluck( $user_task['incomplete'], 'estimation' ) );
                } else {
                    $incomplete_estimation = 0;
                    $total_incomplete_tasks = 0;
                }

                $total_estimation_time = $completed_estimation + $incomplete_estimation;

                if ( empty( $total_estimation_time ) ) {
                    $progress = 0;
                } else {
                    $progress = ($completed_estimation * 100)/$total_estimation_time;
                }

                $user = $users[$user_id];

                $sprint->meta['users'][] = [
                    'total_estimated_time'   => $total_estimation_time,
                    'total_completed_time'   => $completed_estimation,
                    'total_incomplete_time'  => $incomplete_estimation,
                    'progress'               => round( $progress ),
                    'avatar_url'             => get_avatar_url( $user_id ),
                    'display_name'           => $user->display_name,
                    'id'                     => $user_id

                ];
            }
        }

        return $this;
    }

    private function get_sprints_users() {
        global $wpdb;

        $user         = $wpdb->base_prefix . 'users';
        $tb_user_meta = $wpdb->base_prefix . 'usermeta';
        $role_user    = wedevs_pm_tb_prefix() . 'pm_role_user';
        $sprint_projects    = wedevs_pm_tb_prefix() . 'pm_sprint_projects';

        $sprint_ids = is_array( $this->sprint_ids ) ? $this->sprint_ids : [$sprint_ids];
        $sprint_ids = implode( ',', $sprint_ids );

        if ( is_multisite() ) {
            $meta_key = wedevs_pm_user_meta_key();

            $query = "SELECT DISTINCT usr.display_name, usr.user_email, usr.ID as user_id, sp.sprint_id
                FROM {$user} as usr
                    LEFT JOIN {$role_user} as ru ON ru.user_id=usr.ID
                    LEFT JOIN {$tb_user_meta} as umeta ON umeta.user_id = usr.ID
                    LEFT JOIN {$sprint_projects} as sp ON sp.project_id=ru.project_id
                WHERE sp.sprint_id IN ($sprint_ids) AND umeta.meta_key='$meta_key'";

        } else {
            $query = "SELECT DISTINCT usr.display_name, usr.user_email, usr.ID as user_id, sp.sprint_id
                FROM {$user} as usr
                    LEFT JOIN {$role_user} as ru ON ru.user_id=usr.ID
                    LEFT JOIN {$sprint_projects} as sp ON sp.project_id=ru.project_id
                WHERE sp.sprint_id IN ($sprint_ids)";
        }

        $results = $wpdb->get_results( $query );
        $users = [];

        foreach ( $results as $key => $result) {
            $users[$result->sprint_id][$result->user_id] = $result;
        }

        return $users;
    }

    private function get_meta_tb_data() {
        if ( empty( $this->sprint_ids ) ) {
            return $this;
        }
        global $wpdb;
        $metas           = [];
        $tb_projects     = wedevs_pm_tb_prefix() . 'pm_projects';
        $tb_meta         = wedevs_pm_tb_prefix() . 'pm_meta';
        $sprint_format = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data      = $this->sprint_ids;

        $query = "SELECT DISTINCT $tb_meta.meta_key, $tb_meta.meta_value, $tb_meta.entity_id
            FROM $tb_meta
            WHERE $tb_meta.entity_id IN ($sprint_format)
            AND $tb_meta.entity_type = %s ";

        array_push( $query_data, 'sprint' );

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );

        foreach ( $results as $key => $result ) {
            $sprint_id = $result->entity_id;
            unset( $result->entity_id );
            $metas[$sprint_id][] = $result;
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $filter_metas = empty( $metas[$sprint->id] ) ? [] : $metas[$sprint->id];

            foreach ( $filter_metas as $key => $filter_meta ) {
                $sprint->meta[$filter_meta->meta_key] = $filter_meta->meta_value;
            }
        }

        return $this;
    }

    private function total_tasks_count() {
        global $wpdb;
        $metas           = [];
        $tb_tasks        = wedevs_pm_tb_prefix() . 'pm_tasks';
        $tb_boardable    = wedevs_pm_tb_prefix() . 'pm_boardables';
        $sprint_format = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data      = $this->sprint_ids;

        $query ="SELECT DISTINCT count($tb_tasks.id) as task_count, $tb_boardable.board_id as sprint_id FROM $tb_tasks
            LEFT JOIN $tb_boardable  ON $tb_boardable.boardable_id = $tb_tasks.id
            WHERE $tb_boardable.board_id IN ($sprint_format)
            AND $tb_boardable.boardable_type=%s
            AND $tb_boardable.board_type=%s
            group by $tb_boardable.board_id
        ";

        array_push( $query_data, 'task', 'sprint' );

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );

        foreach ( $results as $key => $result ) {
            $sprint_id = $result->sprint_id;
            unset($result->sprint_id);
            $metas[$sprint_id] = $result->task_count;
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $sprint->meta['total_tasks'] = empty( $metas[$sprint->id] ) ? 0 : $metas[$sprint->id];
        }

        return $this;
    }

    private function  total_complete_tasks_count() {
        global $wpdb;
        $metas           = [];
        $tb_tasks        = wedevs_pm_tb_prefix() . 'pm_tasks';
        $tb_boardable    = wedevs_pm_tb_prefix() . 'pm_boardables';
        $sprint_format = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data      = $this->sprint_ids;

        $query ="SELECT DISTINCT count($tb_tasks.id) as task_count, $tb_boardable.board_id as sprint_id FROM $tb_tasks
            LEFT JOIN $tb_boardable  ON $tb_boardable.boardable_id = $tb_tasks.id
            WHERE $tb_boardable.board_id IN ($sprint_format)
            AND $tb_boardable.boardable_type=%s
            AND $tb_boardable.board_type=%s
            AND $tb_tasks.status = %d
            group by $tb_boardable.board_id
        ";

        array_push( $query_data, 'task', 'sprint', '1' );

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );

        foreach ( $results as $key => $result ) {
            $sprint_id = $result->sprint_id;
            unset($result->sprint_id);
            $metas[$sprint_id] = $result->task_count;
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $sprint->meta['total_complete_tasks'] = empty( $metas[$sprint->id] ) ? 0 : $metas[$sprint->id];
        }

        return $this;
    }

    private function total_incomplete_tasks_count() {
        global $wpdb;
        $metas           = [];
        $tb_tasks        = wedevs_pm_tb_prefix() . 'pm_tasks';
        $tb_boardable    = wedevs_pm_tb_prefix() . 'pm_boardables';
        $sprint_format = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data      = $this->sprint_ids;

        $query ="SELECT DISTINCT count($tb_tasks.id) as task_count, $tb_boardable.board_id as sprint_id FROM $tb_tasks
            LEFT JOIN $tb_boardable  ON $tb_boardable.boardable_id = $tb_tasks.id
            WHERE $tb_boardable.board_id IN ($sprint_format)
            AND $tb_boardable.boardable_type=%s
            AND $tb_boardable.board_type=%s
            AND $tb_tasks.status = %d
            group by $tb_boardable.board_id
        ";

        array_push( $query_data, 'task', 'sprint', '0' );

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );

        foreach ( $results as $key => $result ) {
            $sprint_id = $result->sprint_id;
            unset($result->sprint_id);
            $metas[$sprint_id] = $result->task_count;
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $sprint->meta['total_incomplete_tasks'] = empty( $metas[$sprint->id] ) ? 0 : $metas[$sprint->id];
        }

        return $this;
    }

    private function total_comments_count() {
        global $wpdb;
        $metas           = [];
        $tb_pm_comments  = wedevs_pm_tb_prefix() . 'pm_comments';
        $tb_boards       = wedevs_pm_tb_prefix() . 'pm_boards';
        $sprint_format = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data      = $this->sprint_ids;

        $query ="SELECT DISTINCT count($tb_pm_comments.id) as comment_count,
        $tb_boards.id as sprint_id FROM $tb_pm_comments
            LEFT JOIN $tb_boards  ON $tb_boards.id = $tb_pm_comments.commentable_id
            WHERE $tb_boards.id IN ($sprint_format)
            AND $tb_pm_comments.commentable_type = %s
            group by $tb_boards.id
        ";

        array_push( $query_data, 'sprint' );

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );

        foreach ( $results as $key => $result ) {
            $sprint_id = $result->sprint_id;
            unset($result->sprint_id);
            $metas[$sprint_id] = $result->comment_count;
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $sprint->meta['total_comments'] = empty( $metas[$sprint->id] ) ? 0 : $metas[$sprint->id];
        }

        return $this;
    }

    private function total_assignees_count() {
        global $wpdb;
        $metas           = [];
        $tb_users        = $wpdb->base_prefix . 'users';
        $tb_user_meta    = $wpdb->base_prefix . 'usermeta';
        $tb_boardable    = wedevs_pm_tb_prefix() . 'pm_boardables';
        $sprint_format = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data      = $this->sprint_ids;

        if ( is_multisite() ) {
            $meta_key = wedevs_pm_user_meta_key();

            $query ="SELECT DISTINCT count($tb_users.id) as user_count,
                    $tb_boardable.board_id as sprint_id
                FROM $tb_users
                LEFT JOIN $tb_boardable  ON $tb_boardable.boardable_id = $tb_users.id
                LEFT JOIN $tb_user_meta as umeta ON umeta.user_id = $tb_users.ID
                WHERE $tb_boardable.board_id IN ( $sprint_format )
                AND $tb_boardable.board_type = %s
                AND $tb_boardable.boardable_type = %s
                AND umeta.meta_key='$meta_key'
                group by $tb_boardable.board_id
            ";
        } else {
            $query ="SELECT DISTINCT count($tb_users.id) as user_count,
                    $tb_boardable.board_id as sprint_id FROM $tb_users
                LEFT JOIN $tb_boardable  ON $tb_boardable.boardable_id = $tb_users.id
                WHERE $tb_boardable.board_id IN ( $sprint_format )
                AND $tb_boardable.board_type = %s
                AND $tb_boardable.boardable_type = %s
                group by $tb_boardable.board_id
            ";
        }

        array_push( $query_data, 'sprint', 'user' );

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );

        foreach ( $results as $key => $result ) {
            $sprint_id = $result->sprint_id;
            unset($result->sprint_id);
            $metas[$sprint_id] = $result->comment_count;
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $sprint->meta['total_assignees'] = empty( $metas[$sprint->id] ) ? 0 : $metas[$sprint->id];
        }

        return $this;
    }

    private function include_complete_tasks() {
        global $wpdb;
        $with = empty( $this->query_params['with'] ) ? [] : $this->query_params['with'];

        if ( ! is_array( $with ) ) {
            $with = explode( ',', str_replace(' ', '', $with ) );
        }

        $incomplete_tasks = [];

        if ( ! in_array( 'complete_tasks', $with ) ) {
            return $this;
        }

        $tb_tasks        = wedevs_pm_tb_prefix() . 'pm_tasks';
        $tb_spts         = wedevs_pm_tb_prefix() . 'pm_sprint_project_tasks';
        $sprint_format   = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data      = $this->sprint_ids;

        $query ="SELECT DISTINCT $tb_spts.task_id, $tb_spts.sprint_id FROM $tb_spts
            LEFT JOIN $tb_tasks  ON $tb_spts.task_id = $tb_tasks.id
            WHERE $tb_spts.sprint_id IN ($sprint_format)
            AND $tb_tasks.status=%d
        ";

        array_push( $query_data,  1 );

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );

        if( empty($results) ) {
            foreach ( $this->sprints as $key => $sprint ) {
                $sprint->complete_tasks['data'] =  [];
            }

            return $this;
        }

        $task_ids = wp_list_pluck( $results, 'task_id' );

        $tasks = Task::get_results([
            'id' => $task_ids,
            'with' => 'assignees'
        ]);

        $key_tasks = [];
        //$tasks['data'] = count( $task_ids ) == 1 && ! empty( $tasks ) ? [$tasks['data']] : $tasks['data'];

        foreach ( $tasks['data'] as $key => $task ) {
            $key_tasks[$task['id']] = $task;
        }

        foreach ( $results as $key => $result ) {
            if ( empty( $key_tasks[$result->task_id] ) ) {
                continue;
            }

            $complete_tasks[$result->sprint_id][] = $key_tasks[$result->task_id];
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $sprint->complete_tasks['data'] = empty( $complete_tasks[$sprint->id] ) ? [] : $complete_tasks[$sprint->id];
        }

        return $this;
    }

    private function include_incomplete_tasks() {

        global $wpdb;
        $with = empty( $this->query_params['with'] ) ? [] : $this->query_params['with'];

        if ( ! is_array( $with ) ) {
            $with = explode( ',', str_replace(' ', '', $with ) );
        }

        $incomplete_tasks = [];

        if ( ! in_array( 'incomplete_tasks', $with ) ) {
            return $this;
        }

        $tb_tasks        = wedevs_pm_tb_prefix() . 'pm_tasks';
        $tb_spts         = wedevs_pm_tb_prefix() . 'pm_sprint_project_tasks';
        $sprint_format   = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data      = $this->sprint_ids;

        $query ="SELECT DISTINCT $tb_spts.task_id, $tb_spts.sprint_id
            FROM $tb_spts
            LEFT JOIN $tb_tasks  ON $tb_spts.task_id = $tb_tasks.id
            WHERE $tb_spts.sprint_id IN ($sprint_format)
            AND $tb_tasks.status=%d
        ";

        array_push( $query_data,  0 );

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );

        if( empty($results) ) {
            foreach ( $this->sprints as $key => $sprint ) {
                $sprint->incomplete_tasks['data'] =  [];
            }

            return $this;
        }

        $task_ids = wp_list_pluck( $results, 'task_id' );

        $tasks = Task::get_results([
            'id' => $task_ids,
            'with' => 'assignees'
        ]);

        //$tasks['data'] = count( $task_ids ) == 1  && ! empty( $tasks ) ? [$tasks['data']] : $tasks['data'];

        // if ( wedevs_pm_get_estimation_type() == 'subtask' ) {
        //     $task_query_id = implode( ',', $task_ids );

        //     $query ="SELECT sum(estimation) as estimation, parent_id
        //         FROM $tb_tasks
        //         WHERE parent_id IN ( $task_query_id )
        //         GROUP BY parent_id";

        //     $estimations = $wpdb->get_results( $query );
        //     $estimations = wp_list_pluck( $estimations, 'estimation', 'parent_id' );

        //     foreach ( $tasks['data'] as $key => $task ) {
        //         $tasks['data'][$key]['estimation'] = empty( $estimations[ $task['id'] ] ) ? 0 : $estimations[ $task['id'] ]*60;
        //     }
        // }


        $key_tasks = [];

        foreach ( $tasks['data'] as $key => $task ) {
            $key_tasks[$task['id']] = $task;
        }

        foreach ( $results as $key => $result ) {
            if ( empty( $key_tasks[$result->task_id] ) ) {
                continue;
            }

            $incomplete_tasks[$result->sprint_id][] = $key_tasks[$result->task_id];
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $sprint->incomplete_tasks['data'] = empty( $incomplete_tasks[$sprint->id] ) ? [] : $incomplete_tasks[$sprint->id];
        }

        return $this;
    }

    private function include_milestone() {
        global $wpdb;
        $with = empty( $this->query_params['with'] ) ? [] : $this->query_params['with'];

        if ( ! is_array( $with ) ) {
            $with = explode( ',', str_replace(' ', '', $with ) );
        }

        $milestone = [];

        if ( ! in_array( 'milestone', $with ) ) {
            return $this;
        }

        $tb_boards       = wedevs_pm_tb_prefix() . 'pm_boards';
        $tb_boardable    = wedevs_pm_tb_prefix() . 'pm_boardables';
        $sprint_format = wedevs_pm_get_prepare_format( $this->sprint_ids );
        $query_data      = $this->sprint_ids;

        $query ="SELECT DISTINCT $tb_boards.*,$tb_boardable.boardable_id as sprint_id  FROM $tb_boards
                LEFT JOIN $tb_boardable  ON $tb_boardable.board_id = $tb_boards.id
                WHERE $tb_boardable.boardable_id IN ($sprint_format) AND $tb_boards.type=%s " ;

        array_push( $query_data, 'milestone' );

        $results = $wpdb->get_results( $wpdb->prepare( $query, $query_data ) );

        foreach ( $results as $key => $result ) {
            $sprint_id = $result->sprint_id;
            unset($result->sprint_id);
            $milestone[$sprint_id] = $result;
        }

        foreach ( $this->sprints as $key => $sprint ) {
            $sprint->milestone['data'] = empty( $milestone[$sprint->id] ) ? [] : [$milestone[$sprint->id]];
        }

        return $this;
    }

    private function get_selectable_items( $tb, $key ) {
        $select = '';
        $select_items = $this->query_params[$key];

        if ( empty( $select_items ) ) {
            $select = $tb . '.*';
        }

        $select_items = str_replace( ' ', '', $select_items );
        $select_items = explode( ',', $select_items );

        foreach ( $select_items as $key => $item ) {
            $select .= $tb . '.' . $item . ',';
        }

        return substr( $select, 0, -1 );
    }

    private function select() {
        $select = '';

        if ( empty( $this->query_params['select'] ) ) {
            $this->select = $this->tb_sprint . '.*';

            return $this;
        }

        $select_items = $this->query_params['select'];

        if ( ! is_array( $select_items ) ) {
            $select_items = str_replace( ' ', '', $select_items );
            $select_items = explode( ',', $select_items );
        }

        foreach ( $select_items as $key => $item ) {
            $item = str_replace( ' ', '', $item );
            $select .= $this->tb_sprint . '.' . $item . ',';
        }

        $this->select = substr( $select, 0, -1 );

        return $this;
    }

    private function join() {
        return $this;
    }

    private function where() {

        $this->where_id()
            ->where_projects_with_permission()
            ->where_title()
            ->where_status();

        return $this;
    }

    private function where_projects_with_permission() {
        global $wpdb;
        $project_id = isset( $this->query_params['project_id'] ) ? $this->query_params['project_id'] : false;

        // if ( empty( $project_id ) || $project_id == 'false' ) {
        //     return $this;
        // }

        $format = wedevs_pm_get_prepare_format( $project_id );
        $project_id = wedevs_pm_get_prepare_data( $project_id );


        $tb_sprint_projects = wedevs_pm_tb_prefix() . 'pm_sprint_projects';
        $role_user = wedevs_pm_tb_prefix() . 'pm_role_user';

        if ( !wedevs_pm_has_manage_capability() || !empty( $project_id ) ) {
            $this->join .= " LEFT JOIN {$tb_sprint_projects} ON {$this->tb_sprint}.id=$tb_sprint_projects.sprint_id";
        }

        if ( !wedevs_pm_has_manage_capability() ) {
            $this->join .= " LEFT JOIN {$role_user} ON {$role_user}.project_id=$tb_sprint_projects.project_id";
        }

        if ( !empty( $project_id ) ) {
            $this->where .= $wpdb->prepare( " AND {$tb_sprint_projects}.project_id IN ($format)", $project_id );
        }

        if ( !wedevs_pm_has_manage_capability() ) {
            $this->where .= $wpdb->prepare( " AND {$role_user}.user_id IN (%d)", get_current_user_id() );
        }

        return $this;
    }

    /**
     * Filter task by status
     *
     * @return class object
     */
    private function where_status() {
        global $wpdb;
        $status = isset( $this->query_params['status'] ) ? $this->query_params['status'] : false;

        if ( $status === false ) {
            return $this;
        }

        $status = intval( $status );

        // $this->where .= " AND {$this->tb_sprint}.status LIKE '%$status%'";
        $this->where .= $wpdb->prepare( " AND {$this->tb_sprint}.status=%d", $status );

        return $this;
    }

    /**
     * Filter sprint by ID
     *
     * @return class object
     */
    private function where_id() {
        global $wpdb;
        $id = isset( $this->query_params['id'] ) ? $this->query_params['id'] : false;

        if ( empty( $id ) ) {
            return $this;
        }

        if ( is_array( $id ) ) {
            //$query_id = implode( ',', $id );
            $query_format = wedevs_pm_get_prepare_format( $id );
            $this->where .= $wpdb->prepare( " AND {$this->tb_sprint}.id IN ($query_format)", $id );
            // $this->where .= " AND {$this->tb_sprint}.id IN ($query_id)";
        }

        if ( !is_array( $id ) ) {
            // $this->where .= " AND {$this->tb_sprint}.id IN ($id)";
            $this->where .= $wpdb->prepare( " AND {$this->tb_sprint}.id IN (%d)", $id );

            $explode = explode( ',', $id );

            if ( count( $explode ) == 1 ) {
                $this->is_single_query = true;
            }
        }

        return $this;
    }

    /**
     * Filter task by title
     *
     * @return class object
     */
    private function where_title() {
        global $wpdb;
        $title = isset( $this->query_params['title'] ) ? $this->query_params['title'] : false;

        if ( empty( $title ) ) {
            return $this;
        }

        // $this->where .= " AND {$this->tb_sprint}.title LIKE '%$title%'";
        $this->where .= $wpdb->prepare( " AND {$this->tb_sprint}.title LIKE %s", '%'.$title.'%' );

        return $this;
    }

    private function limit() {
        global $wpdb;
        $per_page = isset( $this->query_params['per_page'] ) ? $this->query_params['per_page'] : false;

        if ( $per_page === false || $per_page == '-1' ) {
            return $this;
        }

        // $this->limit = " LIMIT {$this->get_offset()},{$this->get_per_page()}";
        $this->limit = $wpdb->prepare( " LIMIT %d,%d", $this->get_offset(), $this->get_per_page() );

        return $this;
    }

    private function orderby() {
        global $wpdb;

        $tb_pj    = $wpdb->prefix . 'pm_sprints';
        $odr_prms = isset( $this->query_params['orderby'] ) ? $this->query_params['orderby'] : false;

        if ( $odr_prms === false && !is_array( $odr_prms ) ) {
            return $this;
        }

        $orders = [];

        $odr_prms = str_replace( ' ', '', $odr_prms );
        $odr_prms = explode( ',', $odr_prms );

        foreach ( $odr_prms as $key => $orderStr ) {
            $orderStr         = str_replace( ' ', '', $orderStr );
            $orderStr         = explode( ':', $orderStr );
            $orderby          = $orderStr[0];
            $order            = empty( $orderStr[1] ) ? 'asc' : $orderStr[1];
            $orders[$orderby] = $order;
        }

        $order = [];

        foreach ( $orders as $key => $value ) {
            $order[] =  $tb_pj .'.'. $key . ' ' . $value;
        }

        $this->orderby = "ORDER BY " . implode( ', ', $order);

        return $this;
    }

    private function get_offset() {
        $page = isset( $this->query_params['page'] ) ? $this->query_params['page'] : false;

        $page   = empty( $page ) ? 1 : absint( $page );
        $limit  = $this->get_per_page();
        $offset = ( $page - 1 ) * $limit;

        return $offset;
    }

    private function get_per_page() {

        $per_page = isset( $this->query_params['per_page'] ) ? $this->query_params['per_page'] : false;

        if ( ! empty( $per_page ) && intval( $per_page ) ) {
            return intval( $per_page );
        }

        return 2;
    }


    private function get() {
        global $wpdb;
        $id = isset( $this->query_params['id'] ) ? $this->query_params['id'] : false;

        $query = "SELECT SQL_CALC_FOUND_ROWS DISTINCT {$this->tb_sprint}.*
            FROM {$this->tb_sprint}
            {$this->join}
            WHERE %d=%d {$this->where}
            {$this->orderby} {$this->limit} ";

        //echo $wpdb->prepare( $query, 1, 1 ); die();
        $results = $wpdb->get_results( $wpdb->prepare( $query, 1, 1 ) );

        $this->found_rows = $wpdb->get_var( "SELECT FOUND_ROWS()" );
        $this->sprints = $results;

        if ( ! empty( $results ) && is_array( $results ) ) {
            $this->sprint_ids = wp_list_pluck( $results, 'id' );
        }

        if ( ! empty( $results ) && !is_array( $results ) ) {
            $this->sprint_ids = [$results->id];
        }

        return $this;
    }



    private function set_table_name() {
        $this->tb_project          = wedevs_pm_tb_prefix() . 'pm_projects';
        $this->tb_sprint           = wedevs_pm_tb_prefix() . 'pm_sprints';
        $this->tb_task             = wedevs_pm_tb_prefix() . 'pm_tasks';
        $this->tb_project_user     = wedevs_pm_tb_prefix() . 'pm_role_user';
        $this->tb_task_user        = wedevs_pm_tb_prefix() . 'pm_assignees';
        $this->tb_categories       = wedevs_pm_tb_prefix() . 'pm_categories';
        $this->tb_category_project = wedevs_pm_tb_prefix() . 'pm_category_project';
    }
}
